<?php
/*
 * Plugin Name: Simple Cache
 * Description: A lightweight caching plugin for WordPress.
 * Version: 1.0
 * Author: Ryan Briscall
 * Copyright: Dekker Designs, 2025-01-08, All Rights Reserved
*/

if (!defined('ABSPATH')) exit;

class DDSimpleCache {

	private $cache_dir;

	public function __construct() {
		$this->cache_dir = WP_CONTENT_DIR . '/cache/all/';
		add_action('init', [$this, 'serve_cache']);
		add_action('admin_menu', [$this, 'add_admin_menu']);
		add_action('admin_post_dd_clear_all_cache', [$this, 'clear_all_cache']);
		add_action('admin_post_dd_clear_single_cache', [$this, 'clear_cache_file']);
		add_action('admin_post_dd_regenerate_cache', [$this, 'regenerate_all_cache']);
		add_action('admin_post_dd_regenerate_cache_single', [$this, 'regenerate_single_cache']);
		add_action('save_post', [$this, 'clear_post_cache']);
	}

	public function serve_cache() {
		if (is_user_logged_in() || defined('DOING_CRON') || !get_option('dd_cache_enabled', true)) {
			return;
		}
	
		$cache_file = $this->get_cache_file();

		if (file_exists($cache_file)) {
			header('Content-Type: text/html; charset=' . get_option('blog_charset'));
			header('Cache-Control: public, max-age=3600');
			header('Pragma: cache');
			header('Expires: ' . gmdate('D, d M Y H:i:s', time() + 3600) . ' GMT');
			readfile($cache_file);
			exit;
		}

		add_action('shutdown', function() use ($cache_file) {
			if (is_admin() || wp_doing_ajax() || wp_doing_cron()) {
				return;
			}

			global $wp_query;
			if (isset($wp_query->post)) {
				$post = $wp_query->post;
				if ($post && $post->post_status === 'publish') {
					$this->regenerate_cache_for_post($post);
				}
			}
		}, PHP_INT_MAX);
	}

	private function get_cache_file() {
		$url = $_SERVER['REQUEST_URI'];
		return $this->cache_dir . md5($url) . '.html';
	}


	public function clear_post_cache($post_id) {
		$cache_file = $this->get_cache_file();
		if (file_exists($cache_file)) unlink($cache_file);
	}


	public function add_admin_menu() {
		add_menu_page('Simple Cache', 'Simple Cache', 'manage_options', 'dd-simplecache', [$this, 'admin_page']);
	}

	public function admin_page() {
		if (!current_user_can('manage_options')) return;


		if (isset($_GET['cleared']) && $_GET['cleared'] == 'true') {
			echo '<div class="notice notice-success"><p>Cache cleared successfully!</p></div>';
		}

		if (isset($_GET['regenerated']) && $_GET['regenerated'] == 'true') {
			echo '<div class="notice notice-success"><p>Cache regenerated successfully for all posts!</p></div>';
		}

		?>
		<div class="wrap">
			<h1>Simple Cache</h1>
			<form method="post" action="options.php">
				<?php
				settings_fields('dd_simplecache_settings');
				do_settings_sections('dd_simplecache_settings');
				?>
				<table class="form-table">
					<tr>
						<th scope="row"><label for="dd_cache_enabled">Enable Cache</label></th>
						<td><input type="checkbox" id="dd_cache_enabled" name="dd_cache_enabled" value="1" <?php checked(1, get_option('dd_cache_enabled', 1)); ?>></td>
					</tr>
				</table>
				<?php submit_button(); ?>
			</form>
			<hr>
			<h2>Cache Actions</h2>
			<a href="<?php echo admin_url('admin-post.php?action=dd_clear_all_cache'); ?>" class="button button-primary">Clear All Cache</a>
			<a href="<?php echo admin_url('admin-post.php?action=dd_regenerate_cache'); ?>" class="button button-secondary">Regenerate All Cache</a>
			<hr>
			<h2>Cached Pages</h2>
			<?php $this->list_cached_pages(); ?>
		</div>
		<?php
	}

	public function clear_all_cache() {
		array_map('unlink', glob($this->cache_dir . '*.html'));
		wp_redirect(admin_url('admin.php?page=dd-simplecache&cleared=true'));
		exit;
	}

	public function clear_cache_file() {
		if (!isset($_GET['file']) || !current_user_can('manage_options')) {
			wp_die('Unauthorized');
		}
	
		$file = sanitize_text_field($_GET['file']);
		$cache_file = $this->cache_dir . $file . '.html';
	
		if (file_exists($cache_file)) {
			unlink($cache_file);
		}
	
		wp_redirect(admin_url('admin.php?page=dd-simplecache&cleared=true'));
		exit;
	}

	public function regenerate_all_cache() {

		$post_types = get_post_types(['public' => true], 'names');

		$posts = get_posts([
			'post_type'   => $post_types,
			'post_status' => 'publish',
			'numberposts' => -1,
		]);


		foreach ($posts as $post) {
			$this->regenerate_cache_for_post($post);
		}

		wp_redirect(admin_url('admin.php?page=dd-simplecache&regenerated=true'));
		exit;
	}

	public function regenerate_single_cache() {
		if (!isset($_GET['post_id']) || !current_user_can('manage_options')) {
			wp_die('Unauthorized');
		}

		$post_id = intval($_GET['post_id']);
		$post = get_post($post_id);

		if ($post && $post->post_status === 'publish') {
			$this->regenerate_cache_for_post($post);
		}

		wp_redirect(admin_url('admin.php?page=dd-simplecache&regenerated=true'));
		exit;
	}


	private function regenerate_cache_for_post($post) {
		$transient_key = 'regenerating_cache_' . $post->ID;
		if (get_transient($transient_key)) {
			return false;
		}

		set_transient($transient_key, true, 60 * 5);

		$url = get_permalink($post);

		$response = wp_remote_get($url, [
			'timeout' => 10,
			'httpversion' => '1.1',
			'headers' => [
				'User-Agent' => 'WordPress Simple Cache Plugin'
			],
			'sslverify' => false,
		]);

		if (is_wp_error($response)) {
			$error_message = $response->get_error_message();
			error_log('Error regenerating cache for URL: ' . $url . ' | Error: ' . $error_message);
			delete_transient($transient_key);
			return false;
		}

		$status_code = wp_remote_retrieve_response_code($response);
		$content = wp_remote_retrieve_body($response);
	
		if ($status_code != 200) {
			error_log('Failed to fetch URL: ' . $url . ' | HTTP Status: ' . $status_code);
			return false;
		}

		if (!empty($content)) {
			$cache_file = $this->cache_dir . md5(parse_url($url, PHP_URL_PATH)) . '.html';

			if (!file_exists($this->cache_dir)) {
				mkdir($this->cache_dir, 0755, true);
			}

			file_put_contents($cache_file, $content);
		}
		delete_transient($transient_key);
		return true;
	}


	private function list_cached_pages() {
		$post_types = get_post_types(['public' => true], 'names');

		$posts = get_posts([
			'post_type'   => $post_types,
			'post_status' => 'publish',
			'numberposts' => -1,
		]);

		if (empty($posts)) {
			echo '<p>No cached pages found.</p>';
			return;
		}

		echo '<table><tbody>';
		foreach ($posts as $post) {
			$permalink = get_permalink($post);
			$cache_file = $this->cache_dir . md5(parse_url($permalink, PHP_URL_PATH)) . '.html';
			$cache_exists = file_exists($cache_file);
			echo '<tr>';
			echo '<td>';
			if ($cache_exists) {
				echo '<span style="color: green;">(Cached)</span> ';
				echo '<a href="' . admin_url('admin-post.php?action=dd_clear_single_cache&file=' . urlencode(md5(parse_url($permalink, PHP_URL_PATH)))) . '" class="button button-small">Delete Cache</a>';
			} else {
				echo '<span style="color: red;">(Not Cached)</span> ';
			}
			echo '<a href="' . admin_url('admin-post.php?action=dd_regenerate_cache_single&post_id=' . $post->ID) . '" class="button button-small">Regenerate Cache</a>';
			echo '</td><td>';
			echo '<a href="' . esc_url($permalink) . '">' . esc_html($permalink) . '</a> ';
			echo '</td>';
			echo '</tr>';
		}
		echo '</tbody></table>';
	}
}

new DDSimpleCache();

function dd_simplecache_register_settings() {
	register_setting('dd_simplecache_settings', 'dd_cache_enabled');
}
add_action('admin_init', 'dd_simplecache_register_settings');