<?php
/**
 * Plugin Name: Custom Breadcrumbs
 * Description: A plugin to create custom breadcrumb trails for any post type.
 * Version: 1.0
 * License: GPLv2 or later
 * Author: Ryan Briscall
 * Copyright: Dekker Designs, 2025-01-02, All Rights Reserved
 */

if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly.
}

// Register the widget area
function custom_breadcrumbs_register_widget_area() {
	register_sidebar([
		'name'          => __('Breadcrumbs', 'custom-breadcrumbs'),
		'id'            => 'breadcrumbs',
		'description'   => __('A widget area for displaying breadcrumbs.', 'custom-breadcrumbs'),
		'before_widget' => '<div class="breadcrumb-widget">',
		'after_widget'  => '</div>',
	]);
}
add_action('widgets_init', 'custom_breadcrumbs_register_widget_area');

// Create the widget
class Custom_Breadcrumbs_Widget extends WP_Widget {
	public function __construct() {
		parent::__construct(
			'custom_breadcrumbs_widget',
			__('Custom Breadcrumbs', 'custom-breadcrumbs'),
			['description' => __('Displays custom breadcrumbs.', 'custom-breadcrumbs')]
		);
	}

	public function widget($args, $instance) {
		global $post;
		$breadcrumbs = get_option('custom_breadcrumbs');
		$separator = '<span class="separator">' . get_option('custom_breadcrumb_separator', ' &raquo; ') . '</span>';

		if (isset($breadcrumbs[$post->ID])) {
			echo $args['before_widget'];
			echo '<nav class="breadcrumbs">';

			$trail = [];

			// Add the front page as the first crumb
			$front_page_id = get_option('page_on_front');
			if ($front_page_id) {
				$trail[] = '<a href="' . get_permalink($front_page_id) . '">Home</a>';
			}

			foreach ($breadcrumbs[$post->ID] as $crumb) {
				$trail[] = '<a href="' . get_permalink($crumb['post_id']) . '">' . esc_html($crumb['crumb_text']) . '</a>';
			}

			// Add the current post as the last breadcrumb with no link
			$trail[] = esc_html(get_the_title($post->ID));

			echo implode($separator, $trail);
			echo '</nav>';
			echo $args['after_widget'];
		}
	}

	public function form($instance) {
		echo '<p>' . __('Configure breadcrumbs via the Custom Breadcrumbs menu.', 'custom-breadcrumbs') . '</p>';
	}

	public function update($new_instance, $old_instance) {
		return $new_instance;
	}
}
add_action('widgets_init', function() {
	register_widget('Custom_Breadcrumbs_Widget');
});

// Register the admin menu
function custom_breadcrumbs_admin_menu() {
	add_menu_page(
		__('Custom Breadcrumbs', 'custom-breadcrumbs'),
		__('Custom Breadcrumbs', 'custom-breadcrumbs'),
		'manage_options',
		'custom-breadcrumbs',
		'custom_breadcrumbs_admin_page',
		'dashicons-admin-links',
		80
	);
	add_submenu_page(
		'custom-breadcrumbs',
		__('Settings', 'custom-breadcrumbs'),
		__('Settings', 'custom-breadcrumbs'),
		'manage_options',
		'custom-breadcrumbs-settings',
		'custom_breadcrumbs_settings_page'
	);
}
add_action('admin_menu', 'custom_breadcrumbs_admin_menu');

// Admin page for managing breadcrumbs
function custom_breadcrumbs_admin_page() {
	$separator = '<span class="separator">' . get_option('custom_breadcrumb_separator', ' &raquo; ') . '</span>';

	if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['custom_breadcrumbs_nonce'])) {
		if (wp_verify_nonce($_POST['custom_breadcrumbs_nonce'], 'save_custom_breadcrumbs')) {
			$breadcrumbs = get_option('custom_breadcrumbs', []);
			$post_id = intval($_POST['post_id']);
			$crumbs = [];
			foreach ($_POST['crumb_posts'] as $index => $crumb_post_id) {
				$crumbs[] = [
					'post_id' => intval($crumb_post_id),
					'crumb_text' => sanitize_text_field($_POST['crumb_texts'][$index])
				];
			}
			$breadcrumbs[$post_id] = $crumbs;
			update_option('custom_breadcrumbs', $breadcrumbs);
			echo '<div class="updated"><p>' . __('Breadcrumb trail saved.', 'custom-breadcrumbs') . '</p></div>';
		}
	}

	if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['delete_breadcrumbs_nonce'])) {
		if (wp_verify_nonce($_POST['delete_breadcrumbs_nonce'], 'delete_custom_breadcrumbs')) {
			$breadcrumbs = get_option('custom_breadcrumbs', []);
			$delete_post_id = intval($_POST['delete_post_id']);
			unset($breadcrumbs[$delete_post_id]);
			update_option('custom_breadcrumbs', $breadcrumbs);
			echo '<div class="updated"><p>' . __('Breadcrumb trail deleted.', 'custom-breadcrumbs') . '</p></div>';
		}
	}

	$breadcrumbs = get_option('custom_breadcrumbs', []);
	$existing_post_ids = array_keys($breadcrumbs);
	// Group and sort posts by post type
	$post_types = get_post_types(['public' => true], 'objects');
	$posts_by_type = [];
	
	foreach ($post_types as $post_type) {
		$posts = get_posts(['post_type' => $post_type->name, 'numberposts' => -1, 'orderby' => 'title', 'order' => 'ASC']);
		if (!empty($posts)) {
			$posts_by_type[$post_type->name] = $posts;
		}
	}
	
	// Sort post types alphabetically by their labels
	uksort($posts_by_type, function ($a, $b) use ($post_types) {
		return strcmp($post_types[$a]->labels->singular_name, $post_types[$b]->labels->singular_name);
	});
	?>
	<div class="wrap">
		<h1><?php echo __('Custom Breadcrumbs', 'custom-breadcrumbs'); ?></h1>
		<form method="POST">
			<?php wp_nonce_field('save_custom_breadcrumbs', 'custom_breadcrumbs_nonce'); ?>
			<label for="post_id">Select a Post:</label>
			<select name="post_id" id="post_id">
				<?php
				foreach ($posts_by_type as $post_type => $posts) {
					foreach ($posts as $post) {
						if (!in_array($post->ID, $existing_post_ids)) {
							$label = sprintf('(%s) %s', $post_types[$post_type]->labels->singular_name, $post->post_title);
							echo '<option value="' . $post->ID . '">' . esc_html($label) . '</option>';
						}
					}
				}
				?>
			</select>
			<h3><?php echo __('Breadcrumb Trail:', 'custom-breadcrumbs'); ?></h3>
			<div id="crumbs-wrapper">
				<p>
					<select name="crumb_posts[]">
						<option value="">Select a post</option>
						<?php
						foreach ($posts_by_type as $post_type => $posts) {
							foreach ($posts as $post) {
								$label = sprintf('(%s) %s', $post_types[$post_type]->labels->singular_name, $post->post_title);
								echo '<option value="' . $post->ID . '">' . esc_html($label) . '</option>';
							}
						}
						?>
					</select>
					<input type="text" name="crumb_texts[]" placeholder="Custom crumb text" />
				</p>
			</div>
			<button type="button" id="add-crumb">+ Add Crumb</button>
			<br><br>
			<input type="submit" class="button button-primary" value="Save Breadcrumb">
		</form>

		<h2><?php echo __('Existing Breadcrumbs', 'custom-breadcrumbs'); ?></h2>
		<ul id="custom-breadcrumbs-list">
			<?php foreach ($breadcrumbs as $post_id => $crumbs) : ?>
				<li>
					<strong><?php echo get_the_title($post_id); ?>:</strong>
					<?php
					$trail = [];

					// Add the front page as the first crumb
					$front_page_id = get_option('page_on_front');
					if ($front_page_id) {
						$trail[] = '<a href="' . get_permalink($front_page_id) . '">Home</a>';
					}

					foreach ($crumbs as $crumb) {
						$trail[] = '<a href="' . get_permalink($crumb['post_id']) . '">' . esc_html($crumb['crumb_text']) . '</a>';
					}
					$trail[] = esc_html(get_the_title($post_id));
					echo implode($separator, $trail);
					?>
					<form method="POST" style="display:inline;">
						<?php wp_nonce_field('delete_custom_breadcrumbs', 'delete_breadcrumbs_nonce'); ?>
						<input type="hidden" name="delete_post_id" value="<?php echo $post_id; ?>">
						<input type="submit" class="button button-secondary" value="Delete">
					</form>
				</li>
			<?php endforeach; ?>
		</ul>

		<script>
			document.getElementById('add-crumb').addEventListener('click', function() {
			  const wrapper = document.getElementById('crumbs-wrapper');
			  const newCrumb = document.createElement('p');
			  newCrumb.innerHTML = '<select name="crumb_posts[]">' +
				  '<option value="">Select a post</option>' +
				  <?php
				  $options = '';
				  foreach ($posts_by_type as $post_type => $posts) {
					  foreach ($posts as $post) {
						  $label = sprintf('(%s) %s', $post_types[$post_type]->labels->singular_name, $post->post_title);
						  $options .= '<option value="' . $post->ID . '">' . esc_js($label) . '</option>';
					  }
				  }
				  echo json_encode($options);
				  ?> +
				  '</select>' +
				  '<input type="text" name="crumb_texts[]" placeholder="Custom crumb text" />';
			  wrapper.appendChild(newCrumb);
		  });
		</script>
	</div>
	<?php
}

// Settings page for custom breadcrumb separator
function custom_breadcrumbs_settings_page() {
	if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['custom_breadcrumbs_settings_nonce'])) {
		if (wp_verify_nonce($_POST['custom_breadcrumbs_settings_nonce'], 'save_custom_breadcrumbs_settings')) {
			$separator = sanitize_text_field($_POST['breadcrumb_separator']);
			update_option('custom_breadcrumb_separator', $separator);
			echo '<div class="updated"><p>' . __('Settings saved.', 'custom-breadcrumbs') . '</p></div>';
		}
	}

	$separator = get_option('custom_breadcrumb_separator', ' &raquo; ');
	?>
	<div class="wrap">
		<h1><?php echo __('Breadcrumb Settings', 'custom-breadcrumbs'); ?></h1>
		<form method="POST">
			<?php wp_nonce_field('save_custom_breadcrumbs_settings', 'custom_breadcrumbs_settings_nonce'); ?>
			<table class="form-table">
				<tr>
					<th scope="row">
						<label for="breadcrumb_separator">Separator</label>
					</th>
					<td>
						<input type="text" id="breadcrumb_separator" name="breadcrumb_separator" value="<?php echo esc_attr($separator); ?>">
					</td>
				</tr>
			</table>
			<input type="submit" class="button button-primary" value="Save Settings">
		</form>
	</div>
	<?php
}

function custom_breadcrumbs_enqueue_styles() {
	// Frontend styles
	if (!is_admin()) {
		wp_enqueue_style(
			'custom-breadcrumbs-style',
			plugin_dir_url(__FILE__) . 'css/custom-breadcrumbs.css',
			[],
			'1.0'
		);
	}
	
	// Admin styles
	if (is_admin()) {
		wp_enqueue_style(
			'custom-breadcrumbs-admin-style',
			plugin_dir_url(__FILE__) . 'css/custom-breadcrumbs-admin.css',
			[],
			'1.0'
		);
	}
}
add_action('wp_enqueue_scripts', 'custom_breadcrumbs_enqueue_styles');
add_action('admin_enqueue_scripts', 'custom_breadcrumbs_enqueue_styles');

